# -------------------------------------------------------------------
# skyweather2_install.sh
#
# Install script for the SwitchDoc Labs SkyWeather2 system.
# Copyright 2021 sopwith@ismellsmoke.net
# License: Attribution-ShareAlike 4.0 International
#                 (CC BY-SA 4.0)
# https://creativecommons.org/licenses/by-sa/4.0/legalcode
#
# Latest supported SDL version: May 7, 2022 - Version 027.6
#
# Use this script at your own risk! No warranty of any kind provided.
#
# This is a Maker Community provided script.
#    (Not affiliated with SwitchDoc Labs.)
#
# Contact information:
# http://www.ismellsmoke.net
# sopwith@ismellsmoke.net
#
# Version 1.5 - 2022-06-04
# -------------------------------------------------------------------
#! /bin/bash

# Script functions.
exit_error() {
    echo "====================================================================================="
    echo "--- This script did not complete successfully. Check error messages and edit script."
    echo "--- Error message: $1"
    echo "====================================================================================="
    exit 1
}

print_success() {
    echo "============================================="
    echo "|   Install script completed successfully.  |"
    echo "============================================="
    echo ""
    echo "Please reboot the Pi to enable the new settings."
    exit 0
}

echo "============================================="
echo "|      Checking for required files.         |"
echo "============================================="

cd /home/pi
FILE=files.txt
if test -f "$FILE"; then
    :
else
    exit_error "--- Required file files.txt not found."
fi

for REQ_FILE in $(cat files.txt)
do
    if test -f "$REQ_FILE"; then
        :
    else
        exit_error "--- Required file $REQ_FILE not found."
    fi
done

echo "+++ All required files found..."
echo

echo "============================================="
echo "|        Installing required apps.          |"
echo "============================================="

for APP in $(cat apps.txt)
do
    echo ""
    echo "Installing $APP..."
    sudo apt install -y $APP
    if [ $? -ne 0 ]; then
        exit_error "--- Error installing $APP."
    else
        echo "+++ $APP successfully installed."
    fi
    echo ""
done

echo "============================================="
echo "|      Installing required modules.         |"
echo "============================================="

cd /home/pi
for MOD in $(cat modules.txt)
do
    echo ""
    echo "Installing $MOD..."
    sudo pip3 install $MOD
    if [ $? -ne 0 ]; then
        exit_error "--- Error installing $MOD."
    else
        echo "+++ $MOD successfully installed."
    fi
    echo ""
done

echo "============================================="
echo "|         Installing software.              |"
echo "============================================="
echo

echo "Installing rtl-sdr..."
cd /home/pi
if [ -d "rtl-sdr" ]; then
    sudo mv rtl_sdr/ /tmp
fi
git clone git://git.osmocom.org/rtl-sdr.git
if [ $? -ne 0 ]; then
    exit_error "--- Error cloning rtl-sdr.git."
fi
cd rtl-sdr
mkdir build
cd build/
cmake ../ -DINSTALL_UDEV_RULES=ON -DDETACH_KERNEL_DRIVER=ON
if [ $? -ne 0 ]; then
    exit_error "--- Error creating make file for rtl-sdr."
fi
make
if [ $? -ne 0 ]; then
    exit_error "--- Error compiling make file for rtl-sdr."
fi
sudo make install
if [ $? -ne 0 ]; then
    exit_error "--- Error installing rtl-sdr."
else
    echo "rtl-sdr software installed successfully."
fi
sudo ldconfig
echo
echo

echo "Installing rtl-433..."
cd /home/pi
if [ -d "rtl_433" ]; then
    sudo mv rtl_433/ /tmp
fi
git clone https://github.com/switchdoclabs/rtl_433.git
if [ $? -ne 0 ]; then
    exit_error "--- Error cloning rtl-433.git."
fi
cd rtl_433/
mkdir build
cd build/
cmake ..
if [ $? -ne 0 ]; then
    exit_error "--- Error creating make file for rtl-433."
fi
make
if [ $? -ne 0 ]; then
    exit_error "--- Error compiling make file for rtl-433."
fi
sudo make install
if [ $? -ne 0 ]; then
    exit_error "--- Error installing rtl-433."
else
    echo "+++ rtl-433 sotware installed successfully."
fi
sudo ldconfig

echo
echo
echo "Installing SkyWeather2..."
cd /home/pi
if [ -d "SDL_Pi_Skyweather2" ]; then
    sudo mv SDL_Pi_Skyweather2/ /tmp
fi
git clone https://github.com/switchdoclabs/SDL_Pi_Skyweather2.git
if [ $? -ne 0 ]; then
    exit_error "--- Error cloning SDL_Pi_Skyweather2.git."
else
    echo "+++ SkyWeather2 software installed successfully."
fi

echo
echo
echo "Creating SkyWeather2 database..."
cd /home/pi
sudo mysql < SkyWeather2_db_schema.sql
if [ $? -ne 0 ]; then
    exit_error "--- Error creating SkyWeather2 database."
else
    echo "+++ SkyWeather2 database created."
fi

echo
echo
echo "Creating WeatherSenseWireless database..."
cd /home/pi
sudo mysql < WeatherSenseWireless_db_schema.sql
if [ $? -ne 0 ]; then
    exit_error "--- Error creating WeatherSenseWireless database."
else
    echo "+++ WeatherSenseWireless database created."
fi

echo
echo
echo "Updating database to v.27.2 schema..."
cd /home/pi/SDL_Pi_Skyweather2
sudo mysql SkyWeather2 < 27.2.DataBaseUpdate.sql
if [ $? -ne 0 ]; then
    exit_error "--- Error applying v.27.2 database update."
else
    echo "+++ Database schema v.27.2 applied successfully. "
fi

echo
echo
echo "Updating database to v.27.3 schema..."
cd /home/pi/SDL_Pi_Skyweather2
sudo mysql SkyWeather2 < 27.3.DataBaseUpdate.sql
if [ $? -ne 0 ]; then
    exit_error "--- Error applying v.27.3 database update."
else
    echo "+++ Database schema v.27.3 applied successfully. "
fi

echo
echo
echo "Updating WeatherSenseWireless database..."
cd /home/pi/SDL_Pi_Skyweather2
sudo mysql WeatherSenseWireless < WeatherSenseWireless.sql
if [ $? -ne 0 ]; then
    exit_error "--- Error updating WeatherSenseWireless database."
fi
sudo mysql WeatherSenseWireless < updateWeatherSenseWireless.sql
if [ $? -ne 0 ]; then
    exit_error "--- Error updating WeatherSenseWireless."
fi

sed -i 's/no_gutters=True,/className="g-0",/' /home/pi/SDL_Pi_Skyweather2/dash_app/status_page.py

cd /home/pi
sudo chown -R pi:pi .

echo
echo
print_success

