/*---------------------------------------------------------------------------
sop4618_S.cpp
Version 0.5

Written by Sopwith
29 Sep 2014

Redistribution and use in source and binary forms, with or without
modification, is permitted. Use of this software for illegal or
malicious purposed is strictly prohibited. Attribution to the creator(s)
is appreciated but not required.

THIS SOFTWARE IS PROVIDED BY THE CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
------------------------------------------------------------------------------
 Arduino C++ class that extends the capabilities of the ByVac 4618_S class.
 The datasheet for the device can be found here:
 http://www.byvac.co.uk/downloads/datasheets/BV4618%20Datasheet.pdf
------------------------------------------------------------------------------
 Revision history:
 	 Version 0.5
		2014-09-29 - Initial release
------------------------------------------------------------------------------*/
#include <Arduino.h>
#include "sop4618_s.h"

/** Default constructor

\return None
\note If default constructor is called Pins 1,0 and 4x20 LCD will be assumed. 
*/
sop4618_S::sop4618_S() : BV4618_S(1, 0)
{
  init(1, 0);
}

/** Overloaded constructor

\return None
*/
sop4618_S::sop4618_S(uint8_t txPin,            ///< Serial TX pin on Arduino
                     uint8_t rxPin             ///< Serial RX pin on Arduino 
                    ) : BV4618_S(rxPin, txPin)
{
  init(txPin, rxPin);
}

/** Overloaded constructor

\return None
*/
sop4618_S::sop4618_S(uint8_t txPin,    ///< Serial TX pin on Arduino
                     uint8_t rxPin,    ///< Serial RX pin on Arduino
                     uint8_t rows,     ///< Number of rows on LCD
                     uint8_t cols      ///< Number of cols on LCD
                     ) : BV4618_S(rxPin, txPin)
{
  init(txPin, rxPin, rows, cols);
}

// Getter methods
/** Returns the Rx pin

*/
uint8_t sop4618_S::getRXPin() {return m_rxPin;}
	
/** Returns the Tx pin

*/
uint8_t sop4618_S::getTXPin() {return m_txPin;}

/** Returns the LCD line count

*/
uint8_t sop4618_S::getNoRows(){return m_Rows;}

/** Returns the LCD column count

*/
uint8_t sop4618_S::getNoCols() {return m_Cols;}
	
/** Retrieve the Device ID from the LCD

\return void
\note This will not work if the Rx line is not connected
*/
String sop4618_S::getDeviceID()
{
  flush();
  puts("\e[?31d");
  delay(50);
  return readString();
}

/** Retrieve the firmware version from the LCD
\return void
\note This will not work if the Rx line is not connected
*/
String sop4618_S::getVersion()
{
  flush();
  puts("\e[?31f");
  delay(50);
  return readString();
}

/** Show or hide the cursor
\return void
*/
void sop4618_S::cursorShow(bool state)    ///< true - show cursor, false - hide cursor
{
  if(state == true)
  {
    puts("\e[?25h");
    flush();
  }
  else
  {
    puts("\e[?25I");
    flush();
  }
}

/** Toggle backlight
\return void
*/
void sop4618_S::enableBacklight(bool state)  ///< true - turn on backlight, false - turn it off (Default=On)
{
  if(state == true)
  {
    puts("\e[?26h");
  }
  else
  {
    puts("\e[?26I");
  }
  flush(); 
}

/** Move cursor up
\return void
*/
void sop4618_S::cursorUp(uint8_t count)    ///< Number of lines to move up (Default=1)
{
  if(count > m_Rows)
    return;
  cursorMove('A', count);
}

/** Move cursor down
\return void
*/
void sop4618_S::cursorDown(uint8_t count)    ///< Number of lines to move down (Default=1)
{
  if(count > m_Rows)
    return;
  cursorMove('B', count);
}

/**Move cursor right
\return void
*/
void sop4618_S::cursorRight(uint8_t count)    ///< Number of lines to move right (Default=1)
{
  if(count > m_Cols)
    return;
  cursorMove('C', count);
}

/** Move cursor left
\return void
*/
void sop4618_S::cursorLeft(uint8_t count)    ///< Number of lines to move left (Default=1)
{
  if(count > m_Cols)
    return; 
  cursorMove('D', count);
}

/** Move cursor home (1,1)
\return void
*/
void sop4618_S::cursorHome()
{
   puts("\e[H");
   flush();
}

/** Move cursor to X,Y coordinate
\return void
*/
void sop4618_S::cursorXY(uint8_t line,    ///< Line to move to
                         uint8_t col)     ///< Column to move to
{
  if(line < 1 || line > m_Rows)
    return;
  if(col < 1 || col > m_Cols)
    return;
   
  String s("\e[");
  char buf[16];
  char lbuf[4];
  char cbuf[4];
  memset(buf, 0x0, sizeof(buf));
  memset(lbuf, 0x0, sizeof(lbuf));
  memset(cbuf, 0x0, sizeof(cbuf));
  itoa(line, lbuf, 10);
  itoa(col, cbuf, 10);
  s += lbuf + String(';') + cbuf + String('H');
  s.toCharArray(buf, sizeof(buf));
  puts(buf);
  delay(50);
  flush();
}

/** Clear the LCD and move cursor to home position (1,1)
\return void
*/
void sop4618_S::clearLCD()
{
  puts("\e[1E");
  delay(50);
  cursorHome();
  flush();
}

/** Backspace the cursor erasing character
\return void
*/
void sop4618_S::cursorBS(uint8_t count)  ///< Number of columns (Default=1)
{
  if(count < 1 || count > m_Cols)
    return;
    
  for(int x=0;x<count;x++)
    putch('\b');
    delay(10);
    flush();
}

/** Toggle cursor blink
\return void
*/
void sop4618_S::cursorBlink(bool state)  ///< true - cusor blinks, false - cursor is solid (Default=false)
{
  if(state == true)
  {
    puts("\e[15E");
    flush();
  }
  else
  {
    puts("\e[14E");
    flush();
  }
}

/** Retrieve a string of characters from the LCD buffer
\return String containing buffer contents
*/
String sop4618_S::readString()
{
  char buf[20];
  memset(buf,0x0,sizeof(buf));
  for(int x=0;x<20;x++)
 {
   buf[x] = getch();
   if(buf[x] == '*')
   {
    buf[x] = 0x0;
    break; 
   }
 }
   return String(buf);
   flush();
} 

/** Print a String on the LCD
\return void
*/
void sop4618_S::printString(String str)    ///< String of characters to print
{
  if(!str.length())
    return; 
  char buf[80];
  memset(buf,0x0,sizeof(buf));
  str.getBytes((unsigned char*)buf, sizeof(buf));
  puts(buf);
  flush();
}

/**
Print a String on the LCD at X,Y coordinate. 
This is a test
\return void
*/
void sop4618_S::printStringXY(String s,    ///< String to print
                              int row,     ///< Line to start print
                              int col)     ///< Column to start print
{
  cursorXY(row, col);
  printString(s);
}

/** Center a String on the LCD
\return void
*/
void sop4618_S::centerString(String str,    ///< String to print
                             int row)       ///< Line to print the String on
{
   int len = str.length();
   if(!len)
     return;
   if(len > m_Cols)
     str[20] = 0x0;
   int idx = (20-len)/2+1;
   if(idx < 1)
      idx = 1; 
   
   cursorXY(row, idx);
   printString(str);
   flush(); 
}

/** Set the LCD screen size
\return void
*/
void sop4618_S::setScreenSize(uint8_t rows,      ///< No of lines
                              uint8_t cols)      ///< No of columns
{
   if(rows == 0 || cols == 0)
     return;
   
  String s("\e[");
  char rbuf[4];
  char cbuf[4];
  char sbuf[16];
  memset(rbuf, 0x0, sizeof(rbuf));
  memset(cbuf, 0x0, sizeof(cbuf));
  memset(sbuf, 0x0, sizeof(sbuf));
  itoa(rows, rbuf, 10);
  itoa(cols, cbuf, 10);
  s += String(rbuf) + String("L\e[");
  s += String(cbuf) + String("c");
  s.toCharArray(sbuf, sizeof(sbuf));
  puts(sbuf);
  delay(10);
  flush();
}

/**
Shift the LCD display to the right.
\return void
\note Be careful - the content will wrap
*/
void sop4618_S::displayShiftRight()
{
  puts("\e[30E");
  flush();
}

/** Shift the LCD display to the left.
\return void
\note Be careful - the content will wrap
*/
void sop4618_S::displayShiftLeft()
{
  puts("\e[24E");
  flush();  
}

/** Toggle scrolling on/off.
\return void
\note This determines if text that rolls off the bottom row wraps around to row 1
*/
void sop4618_S::displaySetScroll(bool state)  ///< true - scrolling enabled, false - scrolling disabled
{
  if(state)
    puts("\e[0x");  
  else
    puts("\e[1x");
  flush();  
}

/** Protected Init method - sets serial pins and LCD size.
Protected Init method - sets serial pins and LCD size.
\return void
*/
void sop4618_S::init(uint8_t txPin,    ///< Serial TX pin on Arduino
                     uint8_t rxPin,    ///< Serial RX pin on Arduino
                     uint8_t rows,     ///< Number of rows on LCD 
                     uint8_t cols      ///< Number of cols on LCD
                     )
{
  m_rxPin = rxPin;
  m_txPin = txPin;
  m_Rows = rows;	// Default = 4
  m_Cols = cols;	// Default = 20
  begin(9600, 50, '*');
  setScreenSize(m_Rows, m_Cols);
  clearLCD();
  cursorShow(false);
}

/** Sends a raw HD744780U command to LCD.
\return void
\note Direct commands may have unpredictable results.
*/
void sop4618_S::sendCmd(uint8_t cmd)    ///< Command number to send
{
  String s("\e[");
  char buf[12];
  memset(buf, 0x0, sizeof(buf));
  itoa(cmd, buf, 10);
  s += String(buf) + String('E');
  s.toCharArray(buf, sizeof(buf));
  puts(buf);
  flush();
}

/** Protected Move cursor method
\return void
*/
void sop4618_S::cursorMove(char dir,          ///< Direction to move ('A'-up, 'B'-down, 'C'-right, 'D'-left)
                           uint8_t count)     ///< No of positions to move
{
  if(dir < 'A' || dir > 'D')
    return;

  String s("\e[");
  char buf[6];
  memset(buf, 0x0, sizeof(buf));
  itoa(count, buf, 10);
  s += buf + String(dir);
  s.toCharArray(buf, sizeof(buf));
  puts(buf);
  flush();
}

/** Protected method that confirms an ACK was received from the LCD after a command
\return void
\note This determines if text that rolls off the bottom row wraps around to row 1
*/
bool sop4618_S::gotACK()
{
   if('*' == getch())
   {
    gotACK();
     return true;
   }
   return false;
}
 

