#! /usr/bin/env python3

"""---------------------------------------------------------------------------
aosong_am2315.py

Written by Sopwith
February 17, 2019

Redistribution and use in source and binary forms, with or without
modification, are permitted. Use of this software for illegal or
malicious purposed is strictly prohibited. Attribution to the creator(s)
is appreciated but not required.

THIS SOFTWARE IS PROVIDED BY THE CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#-----------------------------------------------------------------------------
 Python application that wraps the capabilities of the AOSONG AM2315 humidity
 and temperature sensor.

 The datasheet for the device can be found here:
 http://www.adafruit.com/datasheets/AM2315.pdf
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
 NOTE: This code is based on the excellent code from Adafruit.
      https://learn.adafruit.com/am2315-encased-i2c-temperature-humidity-sensor/python-circuitpython
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
 Revision history:

     2019-02-17 - Initial release
#--------------------------------------------------------------------------"""
import sys
import math
import time
import argparse
import board
import busio
import adafruit_am2320


class AM2315:
    """Wrapping for an AOSONG AM2315 humidity and temperature sensor.

    Provides simple access to a AM2315 chip using the Adafruit CircuitPython module

    Attributes:
        debug:     bool containing debug state

    """
    def __init__(self, debug=False):

        self.debug = debug  # Debug flag
        self.lastError = ""
        self.i2c = busio.I2C(board.SCL, board.SDA)
        assert self.i2c != None
        self.sensor = adafruit_am2320.AM2320(self.i2c)
        assert self.sensor != None

    def read_sensor_data(self):
        """ Reads the humidity and temperature from the AS2315.

        Args:
            None

        Returns:
          Tuple containing the following fields:
                humidity    - float
                temperature - float (Celsius)
                temperature - float (Fahrenheit)
        """
    	# We read sensor data more than once to be sure we have a successful read.
        t = []
        h = []
        for x in range(12):
            try:
                t.append(self.sensor.temperature)
                h.append(self.sensor.relative_humidity)
                time.sleep(.5)
                t.append(self.sensor.temperature)
                h.append(self.sensor.relative_humidity)
                time.sleep(.5)
                t.append(self.sensor.temperature)
                h.append(self.sensor.relative_humidity)
                if (t[0]+t[1]+t[2])/3 > max(t)-2 and (h[0]+h[1]+h[2])/3 > max(h)-2:
                    break
            except:
                continue

        data = []
        data.append(h[0])
        data.append(t[0])
        f = round((t[0]*9)/5+32,2)
        data.append(f)
        return tuple(data)

    def read_humidity(self):
        """Read humidity data from the sensor.
        Args:
            None

        Returns:
            float = humidity reading, None if error
        """
        return float(self.read_sensor_data()[0])


    def read_temperature(self, fahrenheit=False):
        """Read temperature data from the sensor. (Celsius is default)

        Args:
            bool - if True returns temp in Fahrenheit. Default=False

        Returns:
            float = humidity reading, None if error
        """
        if fahrenheit == False:
            return float(self.read_sensor_data()[1])
        else:
            return  float(self.read_sensor_data()[2])

    def calc_dewpoint(self, temp, rh):
        """Calculate dewpoint from temp and humidity.

        Calculation used in this method was obtained here:
        http://andrew.rsmas.miami.edu/bmcnoldy/Humidity.html

        Algorithm used:
        243.04*(LN(RH/100)+((17.625*T)/(243.04+T)))/(17.625-LN(RH/100)-((17.625*T)/(243.04+T)))

        Args:
            temp: int or float containing C temperature

        Returns:
            Tuple containing C and F dewpoint. None if error.
        """

        dp_C = 243.04 * (math.log(rh * .01) + ((17.625 * temp) / (243.04 + temp))) / (17.625 - math.log(rh * .01) - ((17.625 * temp) / (243.04 + temp)))
        dp_C = math.trunc(dp_C*100) / 100
        dp_F = (dp_C*9)/5 + 32
        dp_F = math.trunc(dp_F*100) / 100
        return (float(dp_C), float(dp_F))

    def get_last_error(self):
        return self.lastError


def GetCmdArgs():

    # Construct the argument parser and parse the arguments
    ap = argparse.ArgumentParser()
    # ap.add_argument("-h", "--help", required=False, help="this help screen")
    ap.add_argument("-f", "--fahrenheit", required=False, action='store_true', help="temp in fahrenheit")
    # Only allow one argument
    group = ap.add_mutually_exclusive_group()
    group.add_argument("-d", "--dewpoint", required=False, action='store_true', help="return dewpoint")
    group.add_argument("-r", "--humidity", required=False, action='store_true', help="return humidity")
    group.add_argument("-t", "--temp", required=False, action='store_true', help="return temperature")


    # Parse command line.
    args = ap.parse_args()
    return args


def main():

    args = GetCmdArgs()
    if args == None:
        sys.exit(0)

    am2315 = AM2315()

    # No args? Just print the data.
    if len(sys.argv) < 2:
        h, c, f = am2315.read_sensor_data()
        print("Temp C:", c)
        print("Temp F:", f)
        print("Humidity:", h)
        dc, df = am2315.calc_dewpoint(c, h)
        print("Dewpoint C:", dc, "- Dewpoint F:", df )
        sys.exit(0)

    if args.humidity == True:
        print(am2315.read_sensor_data()[0])
        sys.exit(0)

    if args.dewpoint == True:
        h, c, f = am2315.read_sensor_data()
        dc, df = am2315.calc_dewpoint(c, h)
        if args.fahrenheit == False:
            print(dc)
        else:
            print(df)
        sys.exit(0)

    if args.temp == True or args.fahrenheit == True:
        c = am2315.read_sensor_data()[1]
        if args.fahrenheit == False:
            print(c)
        else:
            print((c*9)/5 + 32)
        sys.exit(0)

if __name__ == "__main__":

    main()

